<?php
// Check if user is logged in and is pemohon
if (!isLoggedIn() || getUserRole() !== 'pemohon') {
    redirect('?page=login');
}

// Check if user is verified
$stmt = $pdo->prepare("SELECT status_verified FROM pemohon_profiles WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$profile = $stmt->fetch();

if (!$profile || $profile['status_verified'] !== 'verified') {
    setFlashMessage('error', 'Akun Anda belum diverifikasi. Silakan tunggu verifikasi dari petugas.');
    redirect('?page=dashboard');
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nomor_perkara = trim($_POST['nomor_perkara'] ?? '');
    $tahun_perkara = trim($_POST['tahun_perkara'] ?? '');
    $para_pihak = trim($_POST['para_pihak'] ?? '');
    $jenis_perkara = trim($_POST['jenis_perkara'] ?? '');
    $kebutuhan = trim($_POST['kebutuhan'] ?? '');
    $keterangan = trim($_POST['keterangan'] ?? '');
    
    $errors = [];
    
    // Validation
    if (empty($nomor_perkara)) {
        $errors[] = "Nomor perkara harus diisi";
    }
    
    if (empty($tahun_perkara) || !is_numeric($tahun_perkara) || $tahun_perkara < 1900 || $tahun_perkara > date('Y')) {
        $errors[] = "Tahun perkara tidak valid";
    }
    
    if (empty($para_pihak)) {
        $errors[] = "Para pihak harus diisi";
    }
    
    if (empty($jenis_perkara)) {
        $errors[] = "Jenis perkara harus dipilih";
    }
    
    if (empty($kebutuhan)) {
        $errors[] = "Kebutuhan salinan harus dipilih";
    }
    
    // Check if case exists (simplified check)
    if (empty($errors)) {
        // Simplified format validation - more flexible
        if (!preg_match('/^[\d]+\/[A-Za-z\.\-]+\/[\d]{4}\/[A-Za-z\.\-]+$/', $nomor_perkara)) {
            $errors[] = "Format nomor perkara tidak valid. Contoh: 123/Pdt.G/2024/PN.Wtp";
        }
    }
    
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Create directory for file uploads
            $uploadDir = '../writable/uploads/permohonan/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            
            // Generate permohonan code
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM permohonans WHERE YEAR(created_at) = " . date('Y'));
            $count = $stmt->fetch()['total'] + 1;
            $kode_permohonan = 'SP-' . date('Y') . '-' . str_pad($count, 4, '0', STR_PAD_LEFT);
            
            // Insert permohonan
            $stmt = $pdo->prepare("
                INSERT INTO permohonans (
                    pemohon_id, kode_permohonan, nomor_perkara, tahun_perkara, 
                    para_pihak, jenis_perkara, kebutuhan, keterangan, 
                    status, created_at, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW(), NOW())
            ");
            
            $stmt->execute([
                $_SESSION['user_id'],
                $kode_permohonan,
                $nomor_perkara,
                $tahun_perkara,
                $para_pihak,
                $jenis_perkara,
                $kebutuhan,
                $keterangan
            ]);
            
            $permohonanId = $pdo->lastInsertId();
            
            // Handle file uploads
            $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf'];
            $maxFileSize = 5 * 1024 * 1024; // 5MB
            
            if (isset($_FILES['dokumen_pendukung']) && $_FILES['dokumen_pendukung']['error'][0] !== UPLOAD_ERR_NO_FILE) {
                foreach ($_FILES['dokumen_pendukung']['name'] as $key => $name) {
                    if ($_FILES['dokumen_pendukung']['error'][$key] === UPLOAD_ERR_OK) {
                        $tmpName = $_FILES['dokumen_pendukung']['tmp_name'][$key];
                        $size = $_FILES['dokumen_pendukung']['size'][$key];
                        $type = $_FILES['dokumen_pendukung']['type'][$key];
                        
                        // Validate file
                        if (!in_array($type, $allowedTypes)) {
                            throw new Exception("File $name: Format tidak diizinkan");
                        }
                        
                        if ($size > $maxFileSize) {
                            throw new Exception("File $name: Ukuran terlalu besar (max 5MB)");
                        }
                        
                        // Generate unique filename
                        $extension = pathinfo($name, PATHINFO_EXTENSION);
                        $filename = $permohonanId . '_' . $key . '_' . time() . '.' . $extension;
                        $filepath = $uploadDir . $filename;
                        
                        if (move_uploaded_file($tmpName, $filepath)) {
                            // Save file info to database
                            $stmt = $pdo->prepare("
                                INSERT INTO permohonan_files (permohonan_id, file_name, file_path, file_type, file_size, created_at) 
                                VALUES (?, ?, ?, ?, ?, NOW())
                            ");
                            $stmt->execute([$permohonanId, $name, $filename, $type, $size]);
                        } else {
                            throw new Exception("Gagal mengupload file $name");
                        }
                    }
                }
            }
            
            $pdo->commit();
            setFlashMessage('success', "Permohonan berhasil diajukan dengan kode: $kode_permohonan");
            redirect('?page=permohonan-saya');
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = $e->getMessage();
        }
    }
    
    if (!empty($errors)) {
        foreach ($errors as $error) {
            setFlashMessage('error', $error);
        }
    }
}

// Get current year for default
$currentYear = date('Y');
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Permohonan Salinan Putusan - PN Watampone</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        .bg-primary-custom {
            background: linear-gradient(135deg, #1e3a8a, #3b82f6);
        }
        .card-custom {
            border: none;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        .form-floating > label {
            color: #6c757d;
        }
        .form-floating > .form-control:focus ~ label,
        .form-floating > .form-control:not(:placeholder-shown) ~ label {
            color: #0d6efd;
        }
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 2rem;
        }
        .step {
            flex: 1;
            text-align: center;
            position: relative;
        }
        .step.active .step-circle {
            background-color: #0d6efd;
            color: white;
        }
        .step-circle {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: #e9ecef;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            font-weight: bold;
        }
        .required {
            color: #dc3545;
        }
    </style>
</head>
<body class="bg-light">
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary-custom">
        <div class="container">
            <a class="navbar-brand d-flex align-items-center" href="setup.php">
                <img src="/permohonan-salinan-putusan/gambar/Logo_pnwtp.png" alt="PN Watampone" height="35" class="me-2">
                <span>PN Watampone</span>
            </a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="?page=dashboard">
                            <i class="fas fa-tachometer-alt me-1"></i> Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="?page=permohonan-baru">
                            <i class="fas fa-plus-circle me-1"></i> Permohonan Baru
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?page=permohonan-saya">
                            <i class="fas fa-list me-1"></i> Permohonan Saya
                        </a>
                    </li>
                </ul>
                
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i>
                            <?= htmlspecialchars($_SESSION['nama_lengkap']) ?>
                            <span class="badge bg-light text-dark ms-1">Pemohon</span>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="?page=logout">
                                <i class="fas fa-sign-out-alt me-2"></i> Logout
                            </a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <main class="py-4">
        <div class="container">
            <!-- Header -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="d-flex align-items-center justify-content-between">
                        <div>
                            <h1 class="h3 mb-1 text-gray-800">
                                <i class="fas fa-plus-circle me-2 text-primary"></i>
                                Permohonan Salinan Putusan
                            </h1>
                            <p class="text-muted mb-0">Ajukan permohonan salinan putusan pengadilan</p>
                        </div>
                        <div class="text-end">
                            <span class="badge bg-primary px-3 py-2">
                                <i class="fas fa-calendar me-1"></i>
                                <?= date('d F Y') ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Step Indicator -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="step-indicator">
                        <div class="step active">
                            <div class="step-circle">1</div>
                            <div class="step-label">Data Perkara</div>
                        </div>
                        <div class="step">
                            <div class="step-circle">2</div>
                            <div class="step-label">Dokumen</div>
                        </div>
                        <div class="step">
                            <div class="step-circle">3</div>
                            <div class="step-label">Konfirmasi</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Flash Messages -->
            <?php if ($error = getFlashMessage('error')): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle me-2"></i>
                    <?= htmlspecialchars($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <!-- Form -->
            <div class="row justify-content-center">
                <div class="col-lg-10">
                    <div class="card card-custom">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0">
                                <i class="fas fa-file-alt me-2"></i>
                                Form Permohonan Salinan Putusan
                            </h5>
                        </div>
                        <div class="card-body p-4">
                            <form method="POST" enctype="multipart/form-data" id="permohonanForm">
                                <!-- Section 1: Data Perkara -->
                                <div id="section-1">
                                    <h6 class="text-primary mb-3">
                                        <i class="fas fa-gavel me-2"></i>Data Perkara
                                    </h6>
                                    
                                    <div class="row mb-3">
                                        <div class="col-md-8">
                                            <div class="form-floating">
                                                <input type="text" class="form-control" id="nomor_perkara" name="nomor_perkara" 
                                                       placeholder="Nomor Perkara" required 
                                                       value="<?= htmlspecialchars($_POST['nomor_perkara'] ?? '') ?>"
                                                       pattern="[\d]+\/[A-Za-z\.]+\/[\d]{4}\/[A-Za-z\.]+"
                                                       title="Format: 123/Pdt.G/2024/PN.Wtp">
                                                <label for="nomor_perkara">Nomor Perkara <span class="required">*</span></label>
                                            </div>
                                            <div class="form-text">Format: 123/Pdt.G/2024/PN.Wtp</div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-floating">
                                                <input type="number" class="form-control" id="tahun_perkara" name="tahun_perkara" 
                                                       placeholder="Tahun" required min="1900" max="<?= date('Y') ?>"
                                                       value="<?= htmlspecialchars($_POST['tahun_perkara'] ?? $currentYear) ?>">
                                                <label for="tahun_perkara">Tahun Perkara <span class="required">*</span></label>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-12">
                                            <div class="form-floating">
                                                <textarea class="form-control" id="para_pihak" name="para_pihak" 
                                                          placeholder="Para Pihak" style="height: 100px" required><?= htmlspecialchars($_POST['para_pihak'] ?? '') ?></textarea>
                                                <label for="para_pihak">Para Pihak <span class="required">*</span></label>
                                            </div>
                                            <div class="form-text">Contoh: John Doe (Penggugat) vs Jane Smith (Tergugat)</div>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <div class="form-floating">
                                                <select class="form-select" id="jenis_perkara" name="jenis_perkara" required>
                                                    <option value="">Pilih Jenis Perkara</option>
                                                    <option value="Perdata" <?= ($_POST['jenis_perkara'] ?? '') === 'Perdata' ? 'selected' : '' ?>>Perdata</option>
                                                    <option value="Pidana" <?= ($_POST['jenis_perkara'] ?? '') === 'Pidana' ? 'selected' : '' ?>>Pidana</option>
                                                </select>
                                                <label for="jenis_perkara">Jenis Perkara <span class="required">*</span></label>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-floating">
                                                <select class="form-select" id="kebutuhan" name="kebutuhan" required>
                                                    <option value="">Pilih Kebutuhan</option>
                                                    <option value="Salinan Putusan" <?= ($_POST['kebutuhan'] ?? '') === 'Salinan Putusan' ? 'selected' : '' ?>>Salinan Putusan</option>
                                                </select>
                                                <label for="kebutuhan">Kebutuhan Salinan <span class="required">*</span></label>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row mb-4">
                                        <div class="col-12">
                                            <div class="form-floating">
                                                <textarea class="form-control" id="keterangan" name="keterangan" 
                                                          placeholder="Keterangan Tambahan" style="height: 80px"><?= htmlspecialchars($_POST['keterangan'] ?? '') ?></textarea>
                                                <label for="keterangan">Keterangan Tambahan (Opsional)</label>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="d-flex justify-content-end">
                                        <button type="button" class="btn btn-primary" onclick="nextSection(2)">
                                            Selanjutnya <i class="fas fa-arrow-right ms-1"></i>
                                        </button>
                                    </div>
                                </div>

                                <!-- Section 2: Dokumen Pendukung -->
                                <div id="section-2" style="display: none;">
                                    <h6 class="text-primary mb-3">
                                        <i class="fas fa-upload me-2"></i>Dokumen Pendukung
                                    </h6>
                                    
                                    <div class="alert alert-info">
                                        <i class="fas fa-info-circle me-2"></i>
                                        <strong>Informasi:</strong>
                                        <ul class="mb-0 mt-2">
                                            <li>Upload dokumen pendukung (KTP, Surat Kuasa, dll) jika diperlukan</li>
                                            <li>Format yang diizinkan: JPG, PNG, PDF</li>
                                            <li>Ukuran maksimal per file: 5MB</li>
                                            <li>Maksimal 5 file</li>
                                        </ul>
                                    </div>

                                    <div class="mb-4">
                                        <label for="dokumen_pendukung" class="form-label">
                                            <i class="fas fa-paperclip me-2"></i>Dokumen Pendukung (Opsional)
                                        </label>
                                        <input type="file" class="form-control" id="dokumen_pendukung" name="dokumen_pendukung[]" 
                                               multiple accept=".jpg,.jpeg,.png,.pdf">
                                        <div class="form-text">Pilih satu atau lebih file (maksimal 5 file)</div>
                                    </div>

                                    <div id="file-preview" class="mb-4"></div>

                                    <div class="d-flex justify-content-between">
                                        <button type="button" class="btn btn-secondary" onclick="prevSection(1)">
                                            <i class="fas fa-arrow-left me-1"></i> Sebelumnya
                                        </button>
                                        <button type="button" class="btn btn-primary" onclick="nextSection(3)">
                                            Selanjutnya <i class="fas fa-arrow-right ms-1"></i>
                                        </button>
                                    </div>
                                </div>

                                <!-- Section 3: Konfirmasi -->
                                <div id="section-3" style="display: none;">
                                    <h6 class="text-primary mb-3">
                                        <i class="fas fa-check-circle me-2"></i>Konfirmasi Permohonan
                                    </h6>
                                    
                                    <div class="alert alert-warning">
                                        <i class="fas fa-exclamation-triangle me-2"></i>
                                        <strong>Perhatian:</strong> Pastikan semua data yang Anda masukkan sudah benar. 
                                        Setelah permohonan diajukan, data tidak dapat diubah.
                                    </div>

                                    <div id="summary-content" class="mb-4">
                                        <!-- Summary will be populated by JavaScript -->
                                    </div>

                                    <div class="form-check mb-4">
                                        <input class="form-check-input" type="checkbox" id="agreement" required>
                                        <label class="form-check-label" for="agreement">
                                            Saya menyatakan bahwa data yang saya masukkan adalah benar dan saya bertanggung jawab 
                                            penuh atas kebenaran data tersebut. <span class="required">*</span>
                                        </label>
                                    </div>

                                    <div class="d-flex justify-content-between">
                                        <button type="button" class="btn btn-secondary" onclick="prevSection(2)">
                                            <i class="fas fa-arrow-left me-1"></i> Sebelumnya
                                        </button>
                                        <button type="submit" class="btn btn-success" id="submitBtn">
                                            <i class="fas fa-paper-plane me-1"></i> Ajukan Permohonan
                                        </button>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let currentSection = 1;

        function showSection(sectionNum) {
            // Hide all sections
            for (let i = 1; i <= 3; i++) {
                document.getElementById(`section-${i}`).style.display = 'none';
                document.querySelector(`.step:nth-child(${i})`).classList.remove('active');
            }
            
            // Show target section
            document.getElementById(`section-${sectionNum}`).style.display = 'block';
            document.querySelector(`.step:nth-child(${sectionNum})`).classList.add('active');
            
            currentSection = sectionNum;
        }

        function nextSection(sectionNum) {
            if (validateCurrentSection()) {
                if (sectionNum === 3) {
                    generateSummary();
                }
                showSection(sectionNum);
            }
        }

        function prevSection(sectionNum) {
            showSection(sectionNum);
        }

        function validateCurrentSection() {
            if (currentSection === 1) {
                const requiredFields = ['nomor_perkara', 'tahun_perkara', 'para_pihak', 'jenis_perkara', 'kebutuhan'];
                for (let field of requiredFields) {
                    const element = document.getElementById(field);
                    if (!element.value.trim()) {
                        element.focus();
                        alert(`${element.labels[0].textContent.replace(' *', '')} harus diisi`);
                        return false;
                    }
                }
                
                // Validate nomor perkara format
                const nomorPerkara = document.getElementById('nomor_perkara').value;
                const pattern = /^[\d]+\/[A-Za-z\.]+\/[\d]{4}\/[A-Za-z\.]+$/;
                if (!pattern.test(nomorPerkara)) {
                    alert('Format nomor perkara tidak valid. Contoh: 123/Pdt.G/2024/PN.Wtp');
                    document.getElementById('nomor_perkara').focus();
                    return false;
                }
            }
            
            return true;
        }

        function generateSummary() {
            const formData = new FormData(document.getElementById('permohonanForm'));
            const files = document.getElementById('dokumen_pendukung').files;
            
            let summaryHtml = `
                <div class="card">
                    <div class="card-body">
                        <h6 class="card-title">Ringkasan Permohonan</h6>
                        <table class="table table-borderless">
                            <tr>
                                <td width="30%"><strong>Nomor Perkara:</strong></td>
                                <td>${formData.get('nomor_perkara')}</td>
                            </tr>
                            <tr>
                                <td><strong>Tahun Perkara:</strong></td>
                                <td>${formData.get('tahun_perkara')}</td>
                            </tr>
                            <tr>
                                <td><strong>Para Pihak:</strong></td>
                                <td>${formData.get('para_pihak')}</td>
                            </tr>
                            <tr>
                                <td><strong>Jenis Perkara:</strong></td>
                                <td>${formData.get('jenis_perkara')}</td>
                            </tr>
                            <tr>
                                <td><strong>Kebutuhan:</strong></td>
                                <td>${formData.get('kebutuhan')}</td>
                            </tr>`;
            
            if (formData.get('keterangan')) {
                summaryHtml += `
                            <tr>
                                <td><strong>Keterangan:</strong></td>
                                <td>${formData.get('keterangan')}</td>
                            </tr>`;
            }
            
            if (files.length > 0) {
                summaryHtml += `
                            <tr>
                                <td><strong>Dokumen:</strong></td>
                                <td>`;
                for (let file of files) {
                    summaryHtml += `<span class="badge bg-info me-1">${file.name}</span>`;
                }
                summaryHtml += `</td>
                            </tr>`;
            }
            
            summaryHtml += `
                        </table>
                    </div>
                </div>`;
            
            document.getElementById('summary-content').innerHTML = summaryHtml;
        }

        // File upload preview
        document.getElementById('dokumen_pendukung').addEventListener('change', function(e) {
            const files = e.target.files;
            const preview = document.getElementById('file-preview');
            
            if (files.length > 5) {
                alert('Maksimal 5 file yang dapat diupload');
                e.target.value = '';
                return;
            }
            
            preview.innerHTML = '';
            
            if (files.length > 0) {
                let html = '<div class="row">';
                for (let file of files) {
                    const size = (file.size / 1024 / 1024).toFixed(2);
                    html += `
                        <div class="col-md-6 mb-2">
                            <div class="card">
                                <div class="card-body p-3">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-file fa-2x me-3 text-primary"></i>
                                        <div class="flex-grow-1">
                                            <h6 class="mb-1">${file.name}</h6>
                                            <small class="text-muted">${size} MB</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `;
                }
                html += '</div>';
                preview.innerHTML = html;
            }
        });

        // Form submission
        document.getElementById('permohonanForm').addEventListener('submit', function(e) {
            const submitBtn = document.getElementById('submitBtn');
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i> Memproses...';
        });
    </script>
</body>
</html>
