<?php

namespace App\Models;

use CodeIgniter\Model;

class UserModel extends Model
{
    protected $table = 'users';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'username', 
        'email', 
        'password', 
        'role', 
        'is_active', 
        'email_verified_at'
    ];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    protected $validationRules = [
        'username' => 'required|min_length[3]|max_length[100]|is_unique[users.username,id,{id}]',
        'email'    => 'required|valid_email|is_unique[users.email,id,{id}]',
        'password' => 'required|min_length[6]',
        'role'     => 'required|in_list[admin,petugas,pemohon]',
    ];

    protected $validationMessages = [
        'username' => [
            'required'    => 'Username harus diisi',
            'min_length'  => 'Username minimal 3 karakter',
            'max_length'  => 'Username maksimal 100 karakter',
            'is_unique'   => 'Username sudah digunakan',
        ],
        'email' => [
            'required'    => 'Email harus diisi',
            'valid_email' => 'Email tidak valid',
            'is_unique'   => 'Email sudah terdaftar',
        ],
        'password' => [
            'required'   => 'Password harus diisi',
            'min_length' => 'Password minimal 6 karakter',
        ],
        'role' => [
            'required' => 'Role harus dipilih',
            'in_list'  => 'Role tidak valid',
        ],
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    protected $beforeInsert = ['hashPassword'];
    protected $beforeUpdate = ['hashPassword'];

    protected function hashPassword(array $data)
    {
        if (isset($data['data']['password'])) {
            $data['data']['password'] = password_hash($data['data']['password'], PASSWORD_DEFAULT);
        }
        return $data;
    }

    public function getUserWithProfile($id)
    {
        return $this->select('users.*, pemohon_profiles.*')
                    ->join('pemohon_profiles', 'pemohon_profiles.user_id = users.id', 'left')
                    ->where('users.id', $id)
                    ->first();
    }

    public function getPemohonList($status = null)
    {
        $builder = $this->select('users.*, pemohon_profiles.*')
                        ->join('pemohon_profiles', 'pemohon_profiles.user_id = users.id', 'inner')
                        ->where('users.role', 'pemohon');
        
        if ($status) {
            $builder->where('pemohon_profiles.status_verified', $status);
        }
        
        return $builder->findAll();
    }

    public function getUsersByRole($role)
    {
        return $this->where('role', $role)
                    ->where('is_active', true)
                    ->findAll();
    }

    public function verifyLogin($email, $password)
    {
        $user = $this->where('email', $email)
                     ->where('is_active', true)
                     ->first();

        if ($user && password_verify($password, $user['password'])) {
            return $user;
        }

        return false;
    }
}
